package loquebot.drives;

import java.util.ArrayList;
import java.util.Iterator;

import cz.cuni.pogamut.MessageObjects.Item;

import loquebot.Main;
import loquebot.memory.LoqueMemory;
import loquebot.body.LoqueTravel;
import loquebot.body.LoqueSpeech;
import loquebot.body.LoqueNavigator;

/**
 * Responsible for powering-up the bot, where the term <i>powering up</i> means
 * selective foraging of worthy items in order to quickly (i.e. effectively)
 * prepare for combat.
 *
 * <p>Main purpose of this drive is to decide, whether a power-up is necessary
 * and then handle foraging of power-up-suitable items. Since this drive should
 * receive less priority than the {@link LoqueCombat } drive, but more priority
 * than the {@link LoquePursue } drive, the agent automatically prefers healing
 * of severe wounds and hunting for most-wanted items (should he need them),
 * before hunting the enemies.</p>
 *
 * <p>Also, this drive has more priority than the {@link LoqueWander } drive.
 * This results in prioritizing the foraging of the-best-and-most-wanted items
 * before mindlessly forraging the rest of the map.</p>
 *
 * <h4>First weapon</h4>
 *
 * First of all, the agent needs a good weapon. Unless he has a good weapon, no
 * other item can really make a big difference. Well, udamage could, but its
 * usually not easy to get.. Therefore, the drive checks the inventory for good
 * weapon presence (specifically, checks the top score of all present weapons).
 * If there's no such weapon, the drive initiates hunt for a good weapon only.
 * This usually happens only once after each respawn, since the agent does not
 * loose weapons by other means than death.
 *
 * <p>Note: The <i>first weapon</i> decision prevents foraging of other itmes.
 * There might be a question of foraging health as well, when the agent is hurt.
 * Well, thinking about it, having no good weapon to fight with, the health does
 * not really matter in combats, does it? It only prolongs what's imminent.</p>
 *
 * <h4>One good loaded weapon and healing</h4>
 *
 * If the agent already has some good weapons, but none of them is loaded with
 * enough of ammo (usually after survived combats), the agent hunts for ammo
 * into the best weapons he already has, as well as for other good weapons.
 *
 * <p>When the agent is hurt, health packs and vials are added to the list of
 * desired items. It might be questionable, whether nice health is important
 * having no good ammo to fight with, but well, what if the health is along the
 * way to ammo? Since lots of ammo is usually all around the map as well as
 * lots of vials, the agent hunts for whatever is closer.</p>
 *
 * <h4>Further foraging</h4>
 *
 * As the actual inventory score rises, which means the agent already has a good
 * weapon with some nice ammo in it, but the power-up target score has not yet
 * been reached, more and more items are added to the list of desired items.
 *
 * <p><b>Weapons and ammo</b>: The agent selectively prioritizes better weapons
 * and ammo, which he does not have enough of yet. In other words, at first he
 * would forage linkguns, miniguns or rocketry packs only; and only then when
 * he is getting full of those, he would add biorifle or flak as well.</p>
 *
 * <p><b>Health, armor and udamage</b>: When the inventory score rises higher,
 * vials, armors and udamages are added to the list of desires. This is usually
 * satisfactory. Since the agent has some weaponry to fight with, he could also
 * use other benefits. Note that health, armor and udamage are not included in
 * the inventory score. They are treated as bonus.</p>
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoquePowerUp extends LoquePowerUpBase
{
    /**
     * Tells, whether we still want to power up.
     */
    private boolean statusWantToPowerUp = false;

    /*========================================================================*/

    /**
     * Id of the current travel ticket.
     */
    private int travelTicket = -1;

    /*========================================================================*/

    /**
     * Main logic of the power-up. Drives the choosing of good useful items and
     * picking them up.
     *
     * <h4>Cook book</h4>
     * <ul>
     * <li>Check, whether there is still need for powering-up.</li>
     * <li>Choose items, we could use: good weapons we do not carry yet; or
     * some ammo for good weapons we already carry; or maybe even some shield.
     * Of course, the ammo and the shield should have less priority than the
     * weapons, since no combat can be won without a gun to trigger.</li>
     * <li>If we have chosen some good handy items, travel to one of them.</li>
     * </ul>
     *
     * @return True, if the drive decided what shall we do now. False otherwise.
     */
    @Override public boolean doLogic ()
    {
        if (!main.optionPowerUp)
            return false;

        // include power-up logs?
        enableLogs (main.optionPowerUpLogs);

        // do we have an item to travel to?
        if (travelTicket > 0)
        {
            // we have an item, keep traveling..
            if (travel.keepTraveling (travelTicket))
                return true;

            // do not travel anymore..
            log.fine ("PowerUp.doLogic(): done traveling to last item");
            travelTicket = -1;
        }

        // let's choose a new item and start traveling
        if (chooseItem ())
            return true;

        return false;
    }

    /*========================================================================*/

    /**
     * Decides, whether we need some handy weapons/ammo/armor/special to feel
     * happy and courageous. If so, selects worthy pickable items we could use
     * and initiates new travel ticket to one of them. Prioritizes weapons over
     * ammo, when foraging first good weapon.
     *
     * @return True, if we want some item. False, if we're fine already.
     */
    private boolean chooseItem ()
    {
        // what's the total inventory score?
        main.statusInventoryScore = getTotalInventoryScore ();

        // are we good enough?
        if (
            (main.statusInventoryScore < main.optionPowerUpWeaponTarget)
            || (memory.self.getHealth () < main.optionPowerUpHealthTarget)
        )
        {
            // so, we're in need of some handy items
            statusWantToPowerUp = true;

            // any weapons nearby? init travel to one of them..
            if (initTravel (getListOfNearbyWeapons (), true, "clense nearby items"))
                return true;

            // do we have at least one reasonable weapon?
            // FUTURE: well, this could be little tricky on maps with lesser
            // weapon only, since the agent might never try to power-up himself
            // with anything else than this "first good weapon".. no armors..
            // perhaps we should examine the map first and set the limit on the
            // first good weapon accordingly.. or update the weapon info class
            if (getTopInventoryScore() < 10)
            {
                // so, prioritize weapons, since we got nothing..
                if (initTravel (getListOfGoodWeapons (), false, "get first good weapon"))
                    return true;
            }

            // so, with at least one reasonable weapon..
            // get list of the best useful items on the map
            ArrayList<Item> items = getListOfGoodItems ();

            // are we hurt?
            // or do we have a nice inventory score already?
            if (
                (memory.self.getHealth () < 70)
                || (main.statusInventoryScore > 20)
            )
            {
                // add useful health packs
                items.addAll (getListOfGoodHealth ());
            }

            // do we have enough inventory score?
            if (main.statusInventoryScore > 6)
            {
                // add useful armor and udamage pickups
                items.addAll (getListOfGoodArmor ());
                items.addAll (getListOfGoodSpecial ());
            }

            // go get one of the chosen items..
            if (initTravel (items, false, "get more cool stuff"))
                return true;

            log.fine (
                "PowerUp.chooseItems(): could not power-up"
                + ", score " + main.statusInventoryScore
                + ", count " + items.size()
            );
            return false;
        }
        // no need for power-ups
        else
        {
            // if we were not powered up recently..
            if (statusWantToPowerUp)
                // intimidate the enemies..
                speech.speechPoweredUp ();

            // remember the status..
            statusWantToPowerUp = false;
            // and report drive inertia
            return false;
        }
    }

    /*========================================================================*/

    /**
     * Initiates new travel ticket to one of the given items.
     * @param items Items to be traveled to.
     * @param all Whether all items should be foraged.
     * @param msg Log message for this travel.
     * @return Retruns true, if travel initialized.
     */
    private boolean initTravel (ArrayList<Item> items, boolean all, String msg)
    {
        // item pickups are reported slowly, therefore remove items we're
        // standing on, since we won't be able to travel to them anyway..
        Iterator<Item> i = items.iterator ();
        while (i.hasNext ())
        {
            // is it close enough to the agent?
            if (memory.self.getSpaceDistance(i.next ().location) < LoqueNavigator.CLOSE_ENOUGH)
                // then do not add this item..
                i.remove ();
        }

        // no items to run to?
        if (items.size () <= 0)
            return false;

        // log decision info
        log.info (
            "PowerUp.initTravel(): " + msg
            + ", score " + main.statusInventoryScore
            + ", count " + items.size()
        );
        // log chosen items..
        for (Item item : items)
        {
            log.finest (
                "PowerUp.initTravel(): " + item.cls
                + ", ID " + item.ID
                + ", " + item.UnrealID
            );
        }

        // are we going to travel to one of them only?
        if (!all)
        {
            // note: since we do not want the agent to stuck and cycle items
            // in a single room only, we shall travel by random once in a while
            travelTicket = (Math.random () < .2)
                ? travel.initTicketToRandom(items)
                : travel.initTicketToNearest (items);
        }
        // no, we're going to travel to all of them
        else travelTicket = travel.initTicketToAll (items);

        // are we traveling somewhere?
        return (travelTicket > 0);
    }

    /*========================================================================*/

    /** Loque pursue. */
    protected LoqueTravel travel;
    /** Loque speech. */
    protected LoqueSpeech speech;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     * @param travel Loque travel.
     * @param speech Loque speech.
     */
    public LoquePowerUp (Main main, LoqueMemory memory, LoqueTravel travel, LoqueSpeech speech)
    {
        super (main, memory);
        this.travel = travel;
        this.speech = speech;
    }
}